; Foenix Rising Issue #6 (FLASH!) February 2023
;   see accompanying article in /apps.emwhite.org/foenixmarketplace
;   written by Michael Weitman
;
; The code below assembles to a 32K binary which is intended to be loaded at $8000
;   There is also a commented header intended to support .PGX packaging, however,
;   the load address will need to be modified to $7FF8 and of course, SuperBASIC will
;   need to support this (it currently does not).  There are unreleased utilities
;   floating around on the Discord forum, but I've not had the time to investigate.
;
; The upper counter, which starts at 0, counts # of screen 'draws', aka the # of times
;   the 80 x 60 text screen is updated.  The lower counter is the RTC time of day clock
;   which is driven off a 500 ms. interrupt.

.cpu    "w65c02"

* = $0

.fill          	16                  ; reserved for MMU_MEM_CTRL ($00) and MMU_IO_CTRL ($01) plus MMU_MEM_BANK_0 through MMU_MEM_BANK_7
.dsection      	zp                  ; declare section named "zp" aka zero page (from $0; well… from $10 since we reserve 16 bytes)

.section zp
  FROM_PTR    .word ?	; two bytes which will be our low-byte / high-byte pair
  TO_PTR	  .word ?	; same
  BLNKFLG     .byte ?
  BANKCTR     .fill           32	; 32 byte 'data structure'
.send                   	        ; zp (zero-page) section end

*   = $8000
.dsection       codespace

*   = $fffa ; Hardware vectors.
    .word   platform.hw_nmi
    .word   platform.hw_reset
    .word   platform.hw_irq

platform    .namespace
.section codespace

* = $8000
;.byte 'P', 'G', 'X'
;.byte $01
;.byte $00, $00, $80, $00

sprbmp  .binary "Sprite-10numsv2.data",0,16384

* = $E000

hw_nmi:     rti
hw_irq:     rti 

; Memory Management Control
MMU_MEM_CTRL            = $00
MMU_IO_CTRL             = $01
SYS_CTRL_TEXT_PG        = $02
SYS_CTRL_COLOR_PG       = $03

; IO PAGE 0
TEXT_LUT_FG     = $D800
TEXT_LUT_BG	    = $D840

; Interrupt Controller Registers
INT_PEND_0 = $D660  ; Pending register for interrupts 0 - 7
INT_PEND_1 = $D661  ; Pending register for interrupts 8 - 15
INT_MASK_0 = $D66C  ; Mask register for interrupts 0 - 7
INT_MASK_1 = $D66D  ; Mask register for interrupts 8 - 15

; Interrupt bits
INT_VKY_SOF     = $01
INT_VKY_SOL     = $02
INT_PS2_KBD     = $04
INT_PS2_MOUSE   = $08
INT_TIMER_0     = $10
INT_TIMER_1     = $20
INT_RESERVED    = $40
INT_CARTRIDGE   = $80

INT_UART        = $01
;RESERVED       = $02
;RESERVED       = $04
;RESERVED       = $08
INT_RTC         = $10
INT_VIA0        = $20
INT_VIA1        = $40
INT_SDC_INS     = $80

INT_IEC_DATA_i  = $01
INT_IEC_CLK_i   = $02
INT_IEC_ATN_i   = $04
INT_IEC_SREQ_i  = $08
;RESERVED       = $10
;RESERVED       = $20
;RESERVED       = $40
;RESERVED       = $80

; graphics registers
VKY_MSTR_CTRL_0 = $D000
VKY_MSTR_CTRL_1 = $D001

; non-platform constants
OFFSET =    #$80

irqhand pha
	    lda MMU_IO_CTRL
	    pha
	    stz MMU_IO_CTRL
	    lda #INT_RTC	    ; Check for RTC flag
	    bit INT_PEND_1
	    beq return		    ; If it's zero, exit
	    sta INT_PEND_1	    ; Yes: clear flag for RTC 
        lda $d69d
        lda BLNKFLG
        eor #$01
        sta BLNKFLG
        sta $d980
        sta $d988

	    lda $D690		    ; seconds
	    pha			        ; digit
	    and #$f0			; pair
	    lsr a
	    lsr a
    	clc
    	adc OFFSET
     	sta $d95a
    	pla
	    and #$0f
    	asl a
    	asl a
    	clc
    	adc OFFSET
     	sta $d952

    	lda $D692	        ; minutes
    	pha		            ; digit
    	and #$f0		    ; pair
    	lsr a
    	lsr a
    	clc
    	adc OFFSET
     	sta $d96a
    	pla
    	and #$0f
    	asl a
    	asl a
	    clc
    	adc OFFSET
     	sta $d962

        lda $D694           ; hours
        pha                 ; digit
        and #$70            ; pair
        lsr a
        lsr a
        clc
        adc OFFSET
        sta $d97a
        pla
        and #$0f
        asl a
        asl a
        clc
        adc OFFSET
        sta $d972

return:	pla
	    sta MMU_IO_CTRL
	    pla
	    rti

irqreg: sei
        lda #<irqhand
        sta $fffe
        lda #>irqhand
        sta $ffff  ; hw_irq+1
        stz MMU_IO_CTRL

        ; Mask off all but the RTC interrupt
        lda #$ff
        sta INT_MASK_0
        and #~INT_RTC
        sta INT_MASK_1

        ; Clear all pending interrupts
        lda #$ff
        sta INT_PEND_0
        sta INT_PEND_1

        ; Re-enable IRQ handling
        lda #$08            ; UTI is the update flag of the control register
        sta $d69e
        lda #$04            ; enables the PIE feature
        sta $d96c
        lda #$0f            ; updates the RS flag to max of xxxx1111 or 500 ms.
        sta $d69b
        lda #$00            ; reset UTI so no further updates can be made (not convinced that this is necessary)
        sta $d69e           ;   could spend some time testing it but I'd rather spend my time on creative endeavors
        lda $d69d           ; by blindly reading the status bit, we are set for next interrupt
        lda #$01            ; seed the blink flag (which will be xor'd in the handler)
        sta BLNKFLG
        cli
        rts

; each of the four blocks of code updates 2 digits/sprites
; we could do this with a lookup table and a loop in ~1/4 of the space, but it would run more slowly
;   versus this 'fall through' style of coding.

bcdbnk0 lda BANKCTR     ; load bank 0 counter; as noted, this code block repeats for BANKCTR+1, BANKCTR+2, and BANKCTR+3
        pha             ; save a temporarily so we can get it quickly for the low nibble calc; our example is 83 60 01 00
        and #$f0        ; mask off the high nibble only             ; start with  1000 0000 as example (nibble starts with 8)
        lsr a           ; shift right (aka div by 2)                ; right =     0100 0000 nibble is now 4 
        lsr a           ; shift right again to div by 2 again       ; right =     0010 0000 nibble is now 2
        clc             ; prepare to add by clearing the carry bit
        adc #$80        ; add $80, the start of the Sprite bank     ; after add = 1010 0000 or $A0 hex which is the 8th sprite
        sta $d90a       ; store to pointer for the tens digit to mid address of sprite #8 
        pla             ; low nibble retrieve from stack (the unmolested BANKCTR)
        and #$0f        ; mask off the low nibble then do the same as above
        asl a
        asl a
        clc
        adc #$80
        sta $d902       ; store to pointer for the ones digit into mid addr for sprite #8      

        lda BANKCTR+1
        pha          
        and #$f0     
        lsr a        
        lsr a        
        clc          
        adc #$80     
        sta $d91a    
        pla          
        and #$0f     
        asl a
        asl a
        clc
        adc #$80
        sta $d912      

        lda BANKCTR+2  
        pha            
        and #$f0       
        lsr a          
        lsr a          
        clc            
        adc #$80       
        sta $d92a      
        pla            
        and #$0f       
        asl a
        asl a
        clc
        adc #$80
        sta $d922      

        lda BANKCTR+3  
        pha            
        and #$f0       
        lsr a          
        lsr a          
        clc            
        adc #$80       
        sta $d93a      
        pla            
        and #$0f       
        asl a
        asl a
        clc
        adc #$80
        sta $d932      
        rts

telem01 pha             ; preserve a, y, x
        phy
        phx
        asl a
        asl a           ; shift twice to mult index by 4
        tay             ; accum contains bank # offset, transfer to index
        sed             ; set decimal mode
        clc             ; clear carry
        lda BANKCTR,y   ; load low-byte BCD pair otherwise known as (xx,xxx,x99)
        adc #1          ; add 1 ... we are here because a bank was retrieved
        sta BANKCTR,y   ; store back
        iny
        lda BANKCTR,y   ; get next BCD pair (xx,xx9,9xx)
        adc #$00        ; add carry in case the prior went from 99 to 00
        sta BANKCTR,y   ; store it back
        iny
        lda BANKCTR,y   ; same for (xx,99x,xxx)
        adc #$00
        sta BANKCTR,y
        iny
        lda BANKCTR,y   ; same for (99,xxx,xxx)
        adc #$00
        sta BANKCTR,y
        cld
        plx
        ply
        pla
        rts

sprload lda #<sprbmp
        sta FROM_PTR
        lda #>sprbmp
        sta FROM_PTR+1
        lda #$80
        sta TO_PTR+1
        stz TO_PTR
        ldy #$00
sprloop lda (FROM_PTR),y
        sta (TO_PTR),y
        iny
        bne sprloop
        inc FROM_PTR+1
        inc TO_PTR+1
        lda TO_PTR+1
        cmp #$ac
        bne sprloop
        rts

sprdef	stz MMU_IO_CTRL
        lda #$27        ; Sprite enable (bit 20) + Graphics (bit 04) + Overlay text (bit 02) + Text mode (bit 01) 
	    sta VKY_MSTR_CTRL_0
        lda #$00        ; zero aux. graphics flags such as 'font set 0', 'font background color off', 'mon enable', 'double y off', 'double x off', '60 Hz.'
        sta VKY_MSTR_CTRL_1

        ldx #$00        ; zero sprite registers for sprite 0..31
sprzero stz $d900,x
        inx
        bne sprzero

	    ldx #$00
sproutr ldy #$00
sprinnr lda sprdata,x
	    sta $D900,x
	    inx
	    iny
	    cpy #$08		    ; process 1 ‘record'  
	    bne sprinnr
	    lda sprdata,x	    ; pre-check for end
	    cmp #$ff
	    bne sproutr
        rts

hw_reset:                   ; this address is plugged into the reset vector; meaning that execution begins at "main" label below

main    ldx #$ff            ; init stack
        txs
        stz MMU_IO_CTRL     ; default to I/O
        jsr inipal
        jsr irqreg          ; irq registration as described in issue #6
        jsr sprdef          ; define/load the sprite data

        lda #$00            ; initalize the BANKCTR struct with zeros
        ldy #$20
iniloop dey
        sta BANKCTR,y
        bne iniloop

        lda #$02            ; screen memory
        sta MMU_IO_CTRL

scpaint lda #$c0            ; init at $c000
        sta TO_PTR+1
        stz TO_PTR
        ldy #$00

scploop lda (TO_PTR),y
        inc a
        sta (TO_PTR),y
        iny
        bne scploop
        inc TO_PTR+1
        lda TO_PTR+1
        cmp #$d2
        bne scploop

        stz MMU_IO_CTRL
        lda #$00
        jsr telem01         ; call telem routine passing in bank #0 in accumulator         
        jsr bcdbnk0         ; update sprites with latest value

        lda #$02            ; change to I/O block
        sta MMU_IO_CTRL
        jmp scpaint
        
inipal  ldx     #0
paloop  lda     palette,x
        sta     TEXT_LUT_FG,x
        sta     TEXT_LUT_BG,x
        inx
        cpx     #64
        bne     paloop

; Indeed, we load the foregound and background palette above, but this is for chars
;   not sprites.  The 'balls' demo from last month, which this code is based upon
;   used a complex palette, created to accomidate a creamy-green gradient.
;
; The inline code below stuffs color #1 and color #2 (two shades of purple) into the
;   the default graphic LUT.

        lda #$01
        sta MMU_IO_CTRL

        lda #$8b            ; one lovely purple color as #1 (#0 is reserved for transparent, per VICKY)
        sta $D006
        lda #$55
        sta $D005
        lda #$b3
        sta $D004

        lda #$43            ; the other lovely purple color as #2
        sta $D00a
        lda #$08
        sta $D009
        lda #$6f
        sta $D008
        rts

palette     .dword  $000000
            .dword  $ffffff
            .dword  $880000
            .dword  $0000ff    ; was $aaffee
            .dword  $cc44cc
            .dword  $00cc55
            .dword  $00FF00
            .dword  $dddd77
            .dword  $dd8855
            .dword  $664400
            .dword  $ff7777
            .dword  $333333
            .dword  $777777
            .dword  $aaff66
            .dword  $0088ff
            .dword  $bbbbbb

;         __.|sz|sz|lay|lay|lut|lut|enable (see legend)
;        |
;        |      addr-high            x-low
;        |    addr-mid   \          /   x-high
;        | addr-low   \   \        /   /   y-low
;        |______   \   \   \      /   /   /   y-high
;               \   \   \   \    /   /   /   /                                                                      base addr
sprdata .byte $01,$00,$80,$00,$ff,$00,$52,$00	; sprite 0  : ones                                                  $D900 
	    .byte $01,$00,$80,$00,$ed,$00,$52,$00 	; sprite 1  : tens                                                  $D908
	    .byte $01,$00,$80,$00,$db,$00,$52,$00 	; sprite 2  : hundreds                                              $D910
	    .byte $01,$00,$80,$00,$c9,$00,$52,$00 	; sprite 3  : thousands                                             $D918
	    .byte $01,$00,$80,$00,$b7,$00,$52,$00 	; sprite 4  : ten-thousands                                         $D920
	    .byte $01,$00,$80,$00,$a5,$00,$52,$00 	; sprite 5  : hundred-thousands                                     $D928
	    .byte $01,$00,$80,$00,$93,$00,$52,$00 	; sprite 6  : millions                                              $D930
	    .byte $01,$00,$80,$00,$81,$00,$52,$00 	; sprite 7  : ten-millions                                          $D938
	    .byte $01,$00,$bc,$00,$cb,$00,$52,$00 	; sprite 8  : thousands-comma                                       $D940
	    .byte $01,$00,$bc,$00,$95,$00,$52,$00 	; sprite 9  : millions-comma                                        $D948                               
	    .byte $01,$00,$80,$00,$f2,$00,$ba,$00 	; sprite 10 : ones-seconds          byte 0 legend                   $D950
	    .byte $01,$00,$80,$00,$df,$00,$ba,$00 	; sprite 11 : tens-seconds           7 bit - unused                 $D958
	    .byte $01,$00,$80,$00,$c9,$00,$ba,$00 	; sprite 12 : ones-minutes	      6, 5 bit - size (see table 5.2)   $D960
	    .byte $01,$00,$80,$00,$b7,$00,$ba,$00 	; sprite 13 : tens-minutes	      4, 3 bit - layer (0..3)           $D968
	    .byte $01,$00,$80,$00,$a1,$00,$ba,$00 	; sprite 14 : ones-hours		  2, 1 bit - color lut (0..3)       $D970
	    .byte $01,$00,$80,$00,$90,$00,$ba,$00 	; sprite 15 : tens-hours		     0 bit - enable (1 = on)        $D978
	    .byte $01,$00,$a8,$00,$ca,$00,$ba,$00 	; sprite 16 : min/sec-colon                                         $D980
	    .byte $01,$00,$a8,$00,$a2,$00,$ba,$00 	; sprite 17 : hr/min-colon                                          $D988
	    .byte $ff                            	; end-of-data token

.send
.endn