; Foenix Rising - Intermediate Matters - Issue #4 pg. 18 - column 1 - item "d."
;
; This is a fully functioning (though non-interactive) assembly language program developed as an example
; of a simply no-kernal output routine as described in the accompanying article referenced above.
;
; The "Makefile" for this .asm file is embedded within the description on the Foenix Marketplace at /apps/emwhite.org/foenixmarketplace

.cpu    "w65c02"

MASTER_CTRL_REG_L = $D000
Mstr_Ctrl_Text_Mode_En = $01 ; Enable Text Mode
VKY_TXT_CURSOR_CTRL_REG = $D010 ; [0] enable [1..2] flash rate [3] no flash

BORDER_CTRL_REG = $D004 ; Bit[0] - Enable (1 by default) Bit [4..6]: X Scroll Offset (will scroll left) (acceptable value 0..7)
BORDER_COLOR_R
BORDER_COLOR_G
BORDER_COLOR_B

* = $0
.fill           16      ; Reserved
.dsection       zp      ; declare section for zero page (from $0)

* = $0200
.dsection data          ; declare section for data segment (from $0200)

* = $e000
 
.dsection kernel        ; declare section for 'kernel' (from $e00x)
.dsection program       ; declare section for program (follows above)

* = $fffa ; Hardware vectors.
.word   platform.hw_nmi
.word   platform.hw_reset
.word   platform.hw_irq

platform .namespace

.section kernel
; kernel section; of course there is no kernel, so this is our kernel.  This code (the set interrupt flag SEI) begins at $E002
;   and the address of hw_reset is aligned with the reset vector via the .word platform.hw_reset compiler directive above
;
; The reset sequence ends with a jmp to the begining of our small program, label 'start'

hw_nmi
rti
hw_irq
rti
hw_reset:
        ; Clear interrupts (we are not using them in this example)
        sei

        ; Initialize the stack pointer
        ldx #$ff
        txs

        ; Initialize VICKY hardware
        jsr TinyVky_Init

        ; Clear some text
        jsr clear

        ; Default $c000 to general I/O.
        stz $1
        lda #2
        sta $1
        jmp start
     
TinyVky_Init:
        stz $1
        lda #Mstr_Ctrl_Text_Mode_En;
        sta MASTER_CTRL_REG_L
        jsr init_pal
        jsr init_border
        lda #$01
        sta $01
        jsr charload

; We're not using a cursor so turn it off
        stz     VKY_TXT_CURSOR_CTRL_REG
        rts
            
init_border
        stz BORDER_CTRL_REG
        stz BORDER_COLOR_R
        stz BORDER_COLOR_G
        stz BORDER_COLOR_B
        rts

;  Color Palette definitions, code to load the foreground and background look-up-table (LUT), and an 'array' of sorts
; 
; We start with each of the original C64 colors (not Luma adjusted) against a light gray background (15)
;   the first pencolor is black (0) and this is color 0.  The control codes code will switch to any of 8 colors,
;   against the same gray background but... there is an Easter Egg of sorts...
;
;   If you change a color (to purple as example), then reverse field, THEN 'CLR', BG is inherited as the pen and
;   default color.  Now the screen has a purple background.  But a note (this is only temporary) .
;
; Said another way, the backgroud may be changed by choosing a primary color, going RVS, then clearing the screen; but only if you
;   stick to the 'ink' color (which will be gray based on the 'colors' definitions below)
;
; One fun experiment is to choose a set of colors in the 'colors' list, each of which has a different ideal state against different
;   backgrounds.  This will allow you choose the additional colors even though there is not a CTRLCODE directive however the same rules
;   will apply.
;
; The algorithm and the color combinations create a state machine and management of state is controlled via code either directly or as an
;   artifact.  Designers of systems must take this into account and when feasible, leverage the side-effect as a new feature.  When
;   disruptive on the other hand, care must be taken the trap undesired values and correct.

palette
	.dword $000000	;C64 black (BLK)
	.dword $ffffff	;C64 white (WHT)
	.dword $68372b	;C64 red (RED)
	.dword $70a4b2	;C64 cyan (CYN)
	.dword $6f3d86	;C64 purple (PUR)
	.dword $588d43	;C64 green (GRN)
	.dword $352879	;C64 blue (BLU)
	.dword $b8c76f	;C64 yellow (YEL)
	.dword $6f4f25	;C64 orange (ORN)
	.dword $433900	;C64 brown (BRN)
	.dword $9a6759	;C64 pink (PNK)
	.dword $444444	;C64 dark gray (DK GRY)
	.dword $959595	;C64 light gray (LT GRY)
	.dword $9ad284	;C64 light green (LT GRN)
	.dword $6c5eb5	;C64 light blue (LT BLU)
	.dword $6c6c6c	;C64 gray (GRY)

init_pal
        ldx #0
_loop	lda palette,x
		sta TEXT_LUT_FG,x
		sta TEXT_LUT_BG,x
		inx
		cpx #64
		bne _loop
        ldx #$00
        lda colors,x
        sta defcolor
        sta pencolor
		rts

colors
    .byte $0F
    .byte $1F
    .byte $2F
    .byte $3F
    .byte $4F
    .byte $5F
    .byte $6F
    .byte $7F
    .byte $8F ; note that since we are implementing FLASH by settting bit 3 (%00001000); so these values are not valid as far as colors are
    .byte $9F ;   concerned but the IRQ routine (discussed in Issue #5) scans the 8 foreground colors with bit 3 high, and alternates its
    .byte $AF ;   RBG color code in the LUT between the gray background color, and the corresponding lower order foreground color to create
    .byte $BF ;   the flashing effect.  In reality, character memory still contains the text but the foreground color matches the background
    .byte $CF ;   therefore, the character is invisible.
    .byte $DF
    .byte $EF
    .byte $FF

; I/O PAGE 0
TEXT_LUT_FG = $D800
TEXT_LUT_BG	= $D840

; I/O PAGE 2 Text Memory
TEXT_MEM = $C000 	; IO Page 2
COLOR_MEM = $C000 	; IO Page 3

; program variables & constants
; control code constants
BEL = 7
CLR = 17
FLA = 18
RVS = 19
NOR = 21

BLK = 24
WHT = 25
RED = 26
CYN = 27
PUR = 28
GRN = 29
BLU = 30
YEL = 31
.send

.section data ; start
; data 'segment' variables will be dynamically inserted following with the definition above ($0200 onwards)
	    defcolor    .byte ?    ;variable for normal
	    pencolor    .byte ?  	;variable for current color
        x_loc       .byte ?    ;location variable
        from_offset .byte ? ;from print index
        to_offset   .byte ?   ;to print index

.send ; data section end

.section zp ; start
; zero page addresses will be dynamically inserted following the reserved above

FROM_PTR .word ?	    ;zero page 10 and 11
TO_PTR .word ?	    ;zero page 12 and 13

.send ; zp (zero-page) section end

.section program ; start
; this is the main portion of the program; it begins by establishing the x and y coordinates of the string, JSRs to outstrng and then
;   (since there is no monitor or breakpoint set, falls into endless loop, jumping to 'loop')

start	    ldx message		    ;x location (5)
	        ldy message+1	    ;y location (29)
	        lda #<message+2	    ;low byte of string
	        sta FROM_PTR	
	        lda #>message+2	    ;high byte of string
	        sta FROM_PTR+1

            jsr outstrng
loop        jmp loop            ;since we are not defining a BRK vector (aka no monitor), we will just sit here and loop

message	
;   .text   $0A,$09,"Hello, y'all",$00                                                ; simple string
;   .text   $0A,$09,"Please ",$12,"FLASH ME",$15," - thank you, kindly!",$00          ; flash test

    .text   $05,$1D,"8 colors ",BLK,"black ",RVS," reverse ",NOR," ",WHT,"white ",RVS," reverse ",NOR
    .text                       RED,"  red ",RVS," reverse ",NOR,"   ",CYN,"cyan ",RVS," reverse ",NOR,"                "
    .text                       PUR,"purple ",RVS," reverse ",NOR," ",GRN,"green ", RVS," reverse ",NOR
    .text                       BLU," blue ",RVS," reverse ",NOR," ",YEL,"yellow ",RVS," reverse ",$00

; end of example main

outstrng	stx x_loc		    ;save x a variable
	        tya		            ;move y to accum.
 	        asl a		        ;mult x 2 w shift left
	        tay		            ;xfer to y for index
	        lda scrntab+1,y	    ;get high byte of row
	        sta TO_PTR+1
        	lda scrntab,y	    ;get low byte of row
        	adc x_loc		    ;add x value
        	sta TO_PTR
        	lda TO_PTR+1
        	adc #0		        ;take care of carry…
        	sta TO_PTR+1	    ;  just in case
        	stz from_offset
            stz to_offset
txtloop	    ldy from_offset
            lda (FROM_PTR),y	;copy loop
        	beq txtdone		    ;if end of line (null)
        	cmp #$20		    ;compare to “ "
        	bcc ctrlcode	    ;placeholder for now	
        	ldy to_offset
            sta (TO_PTR),y	    ;else store to screen
        	jsr txtcolor	    ;currently an rts
        	inc to_offset       ;bump index
            inc from_offset
        	jmp txtloop
txtdone     rts

ctrlcode	inc from_offset
            cmp #BEL
            beq bell
            cmp #CLR
            beq clrscrn
            cmp #NOR
            beq normal
            cmp #FLA
            beq flash
            cmp #RVS
            beq reverse
            cmp #BLK
            bcc txtloop	;if < lowest (black)
            sec
            sbc #BLK
            tax
            lda colors,x
            sta pencolor	
            jmp txtloop

bell	    jsr playbel
            jmp txtloop

clrscrn     jsr clear
            jmp txtloop

normal      lda defcolor
            sta pencolor
            jmp txtloop

flash       lda pencolor
            ora #$08
            sta pencolor
            jmp txtloop

reverse     lda pencolor
            asl a
            bcc rev_bit1
            ora #%00000001
rev_bit1    asl a
            bcc rev_bit2
            ora #%00000001
rev_bit2    asl a
            bcc rev_bit3
            ora #%00000001
rev_bit3    asl a
            bcc rev_done
            ora #%00000001
rev_done    sta pencolor
            jmp txtloop

txtcolor    lda #$03
            sta $01
            lda pencolor
            ldy to_offset
            sta (TO_PTR),y
            lda #$02
            sta $01
            rts

clear		lda #$20
			ldx #$02
			stx $01
			jsr bytefill
			lda pencolor
			sta defcolor
			ldx #$03
			stx $01
			jsr bytefill
            ldx #$02
            stx $01
			rts

bytefill	ldx #$00			
fillloop	sta $c000,x
			inx
			bne fillloop
			inc fillloop+2
			ldy fillloop+2
			cpy #$d2
			bne fillloop
			ldx #$c0
smfill	    sta $d1ff,x
			dex
    		bne smfill
			ldx #$c0
			stx fillloop+2
			rts

; utility routine to load the custom charaterset identified on line 396 below

charload    lda #<charset
			sta FROM_PTR
			lda #>charset
			sta FROM_PTR+1
			lda #$C0
			sta TO_PTR+1
			stz TO_PTR
			ldy #$00
charloop	lda (FROM_PTR),y
			sta (TO_PTR),y
			iny
			bne charloop
			inc FROM_PTR+1
			inc TO_PTR+1
			lda TO_PTR+1
			cmp #$c8
			bne charloop
            rts

playbel     lda #$00
            tax
beloop1     sta $D400,x
            inx
            cpx #$17	; init most of the regs
            bne beloop1
            lda #$32	; ~5th octave G
            sta $D401	;   stuffed in freq hi byte
            lda #$69
            sta $D405	; attack / decay
            lda #$8A
            sta $D406	; sustain / release	
            lda #$4C
            sta $D418	; hi-pass / volume (12)
            lda #$60
            sta $D416	; filt cutoff (high)
            lda #$11
            sta $D417	; resonance control reg
            lda #$11
            sta $D404	; triangle / gate-on (“key on”)
            ldy #$00	; useless delay loop...
            ldx #$00	;   useless because it merely
beloop2     inx			;   wastes cycles, iterating 
            bne beloop2 ;   32 * 256 = 8192 times
            iny
            cpy #$20	; at varied clock speeds
            bne beloop2	; this will no longer work
            lda #$10
            sta $D404	; “key off”
            rts 

charset .binary "f256jr_std-charset.bin",0,2048

; this block of .word directives defines starting addresses of an 80 x 60 screen
;   if you are working with an 80 x 50 screen, no problem
;   but if you are working with a 40 column screen, you'll need to adjust by inserting
;   lines at the midpoint between each (e.g. $C000, $C028, $C050, $C078, $C0A0, etc.)

scrntab 
    .word $C000
    .word $C050
    .word $C0A0
    .word $C0F0
    .word $C140
    .word $C190
    .word $C1E0
    .word $C230
    .word $C280
    .word $C2D0
    .word $C320
    .word $C370
    .word $C3C0
    .word $C410
    .word $C460
    .word $C4B0
    .word $C500
    .word $C550
    .word $C5A0
    .word $C5F0
    .word $C640
    .word $C690
    .word $C6E0
    .word $C730
    .word $C780
    .word $C7D0
    .word $C820
    .word $C870
    .word $C8C0
    .word $C910
    .word $C960
    .word $C9B0
    .word $CA00
    .word $CA50
    .word $CAA0
    .word $CAF0
    .word $CB40
    .word $CB90
    .word $CBE0
    .word $CC30
    .word $CC80	  
    .word $CCD0	  
    .word $CD20	  
    .word $CD70	  
    .word $CDC0
    .word $CE10	  
    .word $CE60	  
    .word $CEB0	  
    .word $CF00	  
    .word $CF50 ; Last line if only 50 line screen
    .word $CFA0
    .word $CFF0
    .word $D040
    .word $D090
    .word $D0E0
    .word $D130
    .word $D180
    .word $D1D0
    .word $D220
    .word $D270

.send         ; program section end
        .endn ; namespace end